package loquebot.util;

import java.util.ArrayList;

import cz.cuni.pogamut.MessageObjects.AddWeapon;
import cz.cuni.pogamut.MessageObjects.ItemType;

/**
 * Enum of weapons with their valuable info. These values are used to tweak the
 * bot's behaviour based on weapons he carries and sees.
 *
 * <p>Note: Most of the values here were empirically settled on many combat
 * results. Therefore, the values are strongly dependent on this bot behaviour
 * and logic. You are welcome to try to use them for other bots, but be aware
 * that they do not represent real values from <i>UT.XEngine</i>.</p>
 *
 * <p>Also note that some mutators  (e.g. berserk, rulez modifier) can change
 * game rules (speed of projectiles, damage dealt, splash radius) and these
 * values would then no longer apply. All of these were settled empirically by
 * <i>brute-forcing</i> different values in many testing combats.</p>
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public enum LoqueWeaponInfo
{
    /*========================================================================*/

    SHIELD_GUN ("XWeapons.ShieldGun", "hammer")
    {
        @Override protected void fill ()
        {
            // this is a minor weapon with no ammo
            minorWeapon = true;
            pickupWeaponAmount = 0;
            pickupAmmoAmount = 0;
            priLowAmmo = altLowAmmo = -1;
            // the shield gun can only be used up-close and only in primary
            // fire mode since the alternate fire mode does not deal damage
            priIdealCombatRange = 0;
            priStrafingAmount = 0;
            priStaticAimAhead = 20;
            priKeepShooting = true;
        }

        @Override public boolean altFire (double distance)
        {
            // no alternate fire mode is available, since it is a shield
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // shield gun is not really a weapon is it?
            return 0;
        }

        @Override public int generalScore (int skill)
        {
            // shield gun is not really a weapon is it?
            return 0;
        }

        @Override public int combatScore (int skill, double distance)
        {
            // shield gun is not really a weapon is it?
            return 0;
        }
    },
    ASSAULT_RIFLE ("XWeapons.AssaultRifle", "handgun")
    {
        @Override protected void fill ()
        {
            // this is a minor weapon
            minorWeapon = true;
            pickupWeaponAmount = 100;
            pickupAmmoAmount = 50;
            // ammo subsides quickly
            priLowAmmo = altLowAmmo = 15;
            // primary fire mode fires speeding bullets
            // alternate fire mode spawns small projectiles, however,
            // we do not use alternate fire mode due to ammo issues
        }

        @Override public boolean altFire (double distance)
        {
            // aternate fire mode launches small grenades, like flak cannon
            // however, current bot does not have access to the actual count
            // of these grenades, so we can not allow alternate fire mode
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // skill level has minimum effect on the score
            return (ammo > 30) ? 1 : 0;
        }

        @Override public int generalScore (int skill)
        {
            return (int) (564 - 0.84 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // distance does not affect the score
            return generalScore(skill);
        }
    },
    BIO_RIFLE ("XWeapons.BioRifle", "bio shit")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 20;
            pickupAmmoAmount = 20;
            // both primary and alternate fire mode spawn the same type of
            // projectiles, therefore the values are the same
            priProjectileSpeed = altProjectileSpeed = 2000;
            priMaxRange = altMaxRange = 2000;
            priEffectiveDistance = altEffectiveDistance = 1200;
            priStaticAimAhead = altStaticAimAhead = 50;
            priStaticZCorrection = altStaticZCorrection = -10;
            priHelpAim = altHelpAim = false;
        }

        @Override public boolean altFire (double distance)
        {
            // using alternate fire mode (slowly pulling bigger blobs) are not
            // easily supported by bots because of timming issues, etc.
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // to much ammo should not always help the score
            if (skill >= 3)
                ammo = Math.min(ammo, 20);
            // skill level affects(decreases) comparable score a lot
            return
                (skill <= 2)
                ? (int) (ammo / 2.0)
                : (int) (ammo / (4.5 + skill));
        }

        @Override public int generalScore (int skill)
        {
            return (int) (1000 - 82.55 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // consider maximum distance..
            if (distance > this.priMaxRange)
                return 0;

            // otherwise, the distance does not affect the score
            return generalScore(skill);
        }
    },
    SHOCK_RIFLE ("XWeapons.ShockRifle", "purple laser")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 20;
            pickupAmmoAmount = 10;
            // primary fire mode works as speeding bullets
            // alternate fire mode works as projectiles
            altProjectileSpeed = 1150;
            altSplashRadius = 150;
            altEffectiveDistance = 700;
            altHelpAim = false;
        }

        @Override public boolean altFire (double distance)
        {
            // we do not use alternate fir mode since it is not worth it
            // alternate/primary combos (which result in nice splash damage)
            // are not used (due to issues with timming and calculations)
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // to much ammo should not always help the score
            // skill level affects(increases) comparable score a lot
            return (skill <= 4)
                ? (int) (Math.min(ammo, 20) / (9.0 + (4 - skill)))
                : (int) (ammo / ((skill == 5) ? 2.0 : 1.6));
        }

        @Override public int generalScore (int skill)
        {
            return (int) (136 + 123.31 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // distance does not affect the score
            return generalScore(skill);
        }
    },
    MINIGUN ("XWeapons.Minigun", "mini-gun")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 150;
            pickupAmmoAmount = 50;
            // ammo subsides quickly
            priLowAmmo = altLowAmmo = 30;
            // both primary and alternate modes shoot speeding bullets
        }

        @Override public boolean altFire (double distance)
        {
            // primary fire mode seems to be more effective for bots
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // to much ammo should not always help the score
            // skill level affects comparable score a bit
            return (skill <= 2)
                ? (int) (Math.min(ammo, 150) / 15.0)
                : (int) (ammo / ((skill == 5) ? 11.0 : 12.0));
        }

        @Override public int generalScore (int skill)
        {
            return (int) (880 - 24.66 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // distance does not affect the score
            return generalScore(skill);
        }
    },
    LINK_GUN ("XWeapons.LinkGun", "link shot")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 70;
            pickupAmmoAmount = 50;
            // ammo subsides quickly
            priLowAmmo = altLowAmmo = 10;
            // primary fire mode spawns projectiles
            priProjectileSpeed = 1000;
            priEffectiveDistance = 1200;
            priStaticAimAhead = 0;
            priHelpAim = false;
            // alternate fire mode works like speeding bullets with range limit
            altMaxRange = 1100;
            altEffectiveDistance = 1100;
        }

        @Override public boolean altFire (double distance)
        {
            // the alternate fire mode is not usually worth it
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // skill level affects comparable score a bit
            return (int) (ammo / ((skill == 7) ? 7.0 : 6.0));
        }

        @Override public int generalScore (int skill)
        {
            return (int) (1174 - 61.52 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // the distance does not affect the score
            return generalScore(skill);
        }
    },
    FLAK_CANNON ("XWeapons.FlakCannon", "flak shrapnel")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 15;
            pickupAmmoAmount = 10;
            // primary fire mode spawns fast projectiles
            priProjectileSpeed = 1900;
            priMaxRange = 4000;
            priIdealCombatRange = 400;
            priEffectiveDistance = 2000;
            priStaticAimAhead = 0;
            priHelpAim = false;
            // alternate fire mode spawns slower projectiles
            altProjectileSpeed = 1200;
            altMaxRange = 1700;
            altSplashRadius = 220;
            altIdealCombatRange = 400;
            altEffectiveDistance = 700;
            altStaticAimAhead = 50;
            altStaticZCorrection = -10;
            altHelpAim = false;
        }

        @Override public boolean altFire (double distance)
        {
            // the alternate fire mode is quite difficult to use and handle
            // since it's terrain/obstacle dependent and also quite slow
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // skill level affects(decreases) comparable score a lot
            return
                (skill >= 5)
                ? (int) (Math.min(ammo, 15) / 5.5)
                : (int) (ammo / 1.4);
        }

        @Override public int generalScore (int skill)
        {
            return (int) (958 - 63.80 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // consider maximum distance..
            if (distance > this.priMaxRange)
                return 0;

            // otherwise, the distance does not affect the score
            return generalScore(skill);
        }
    },
    ROCKET_LAUNCHER ("XWeapons.RocketLauncher", "rocketry")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 12;
            pickupAmmoAmount = 9;
            // both primary and alternate fire mode spawn the same type of
            // projectiles, therefore the values are the same
            priProjectileSpeed = altProjectileSpeed = 1300;
            priSplashRadius = altSplashRadius = 220;
            priIdealCombatRange = altIdealCombatRange = 400;
            priEffectiveDistance = altEffectiveDistance = 800;
            priStaticAimAhead = altStaticAimAhead = 50;
            priStaticZCorrection = altStaticZCorrection = -50;
            priHelpAim = altHelpAim = false;
        }

        @Override public boolean altFire (double distance)
        {
            // alternate fire mode causes lots of suicides, nothing else,
            // thus primary fire mode seems to be more effective for now
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // skill level affects(decreases) comparable score a lot
            return
                (skill >= 6)
                ? (int) (Math.min(ammo, 12) / (5.5 + (skill - 6)))
                : (int) (ammo / ((skill == 5) ? 1.2 : 1));
        }

        @Override public int generalScore (int skill)
        {
            return (int) (1251 - 107.83 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // the distance does not affect the score
            return generalScore(skill);
        }
    },
    LIGHTNING_GUN ("XWeapons.SniperRifle", "piece of lightning")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 15;
            pickupAmmoAmount = 10;
            // primary fire mode shoots speeding bullets, aim for headshots
            // alternate fire mode activates zoom
            priStaticZCorrection = 40;
        }

        @Override public boolean altFire (double distance)
        {
            // second fire mode activates zoom
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // to much ammo should never help the score
            ammo = Math.min(ammo, 15);
            // skill level affects(increases) comparable score a lot
            return
                (skill >= 6)
                ? (int) (ammo / 4.0)
                : ((ammo > 5) ? 1 : 0);
        }

        @Override public int generalScore (int skill)
        {
            return (int) (54 + 91.60 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // distance does not affect the score
            return generalScore(skill);
        }
    },
    SNIPER_RIFLE ("UTClassic.ClassicSniperRifle", "classic one")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 15;
            pickupAmmoAmount = 10;
            // primary fire mode shoots speeding bullets, aim for headshots
            // alternate fire mode activates zoom
            priStaticZCorrection = 40;
        }

        @Override public boolean altFire (double distance)
        {
            // second fire mode activates zoom
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // to much ammo should never help the score
            // skill level affects(increases) comparable score a lot
            return
                (skill <= 5)
                ? ((ammo > 5) ? 1 : 0)
                : (int) (ammo / (1.4 - (skill - 6) * 0.1));
        }

        @Override public int generalScore (int skill)
        {
            return (int) (20 + 116.70 * skill);
        }

        @Override public int combatScore (int skill, double distance)
        {
            // distance does not affect the score
            return generalScore(skill);
        }
    },
    REDEEMER ("XWeapons.Redeemer", "RE-DEEM-ER")
    {
        @Override protected void fill ()
        {
            pickupWeaponAmount = 1;
            pickupAmmoAmount = 1;
            // primary fire mode spawns projectiles with big splash radius
            priProjectileSpeed = 1000;
            priSplashRadius = 500; // original value is 2000
            priHelpAim = false;
            // alternate fire mode is not supported for bots
        }

        @Override public boolean altFire (double distance)
        {
            // alternate fire mode is not supported for bots
            return false;
        }

        @Override public int inventoryScore (int skill, int ammo)
        {
            // go fighting, if we have a redeemer! kill something asap!
            return 999999;
        }

        @Override public int generalScore (int skill)
        {
            // you're kidding me? it's called "re-deem-er"
            return 999999;
        }

        @Override public int combatScore (int skill, double distance)
        {
            // you're kidding me? it's called "re-deem-er"
            return 999999;
        }
    },
    // and a fail-safe value..
    UNKNOWN ("???", "gun")
    {
        @Override protected void fill () {}
        @Override public boolean altFire (double distance) { return false; }
        @Override public int inventoryScore (int skill, int ammo) { return 0; }
        @Override public int generalScore (int skill) { return 0; }
        @Override public int combatScore (int skill, double distance) { return 0; }
    };

    /*========================================================================*/

    /** Name of the weapon in UT engine. */
    public String name;

    /** Name of the weapon in speech. */
    public String title;

    /**
     * Whether this weapon is a minor one. Should the bot be fighting with this
     * weapon, it should try to pick something better ASAP.
     */
    public boolean minorWeapon = false;

    /**
     * How much ammo this weapon adds upon pickup.
     */
    public int pickupWeaponAmount = 0;
    /**
     * How much ammo this weapon adds upon pickup.
     */
    public int pickupAmmoAmount = 0;

    /**
     * Projectile speed of primary fire mode.
     * Defaults to value 9e+9 for speeding bullets.
     */
    public double priProjectileSpeed = 9e+9;
    /**
     * Max distance to which the weapon can shoot with primary fire mode.
     * Defaults to 9e+9, when there is no range limit.
     */
    public double priMaxRange = 9e+9;
    /**
     * Splash damage radius of primary fire mode. Zero if no splash damage.
     */
    public double priSplashRadius = 0;
    /**
     * Whether the weapon is better to be charged, even if no enemy is around.
     */
    public boolean priKeepShooting = false;

    /**
     * Amount of ammo, when the weapon is to be considered empty.
     */
    public int priLowAmmo = 3;

    /**
     * Most effective distance from the enemy while using primary fire mode.
     * This value is used to keep desired and working distance from enemies.
     */
    public double priIdealCombatRange = 300;
    /**
     * Most effective amount of strafing to side.
     */
    public double priStrafingAmount = 400;
    /**
     * Effective distance of primary fire mode. Beyond this distance, the
     * weapon might start to loose some glance. This value is used as cut-off
     * point for aim-ahead predictions based on enemy distance.
     */
    public double priEffectiveDistance = 9e+9;
    /**
     * How much time it is reasonable to let the projectile of primary fire
     * mode glide until expected impact. This value is used as cut-off point
     * for aim-ahead predictions based on enemy velocity and distance.
     */
    public double priReasonableFlight = .8;
    /**
     * How much static aim-ahead distance should be used for primary fire
     * mode. This value is used for aim-ahead predictions of speeding
     * bullets.
     */
    public double priStaticAimAhead = 60;
    /**
     * How much static z-axis correction should be used for primary fire
     * mode. This value is used for headshots and predictions.
     */
    public double priStaticZCorrection = 0;

    /**
     * Whether server may assist with aim corrections for primary fire mode.
     * This is reasonable only for speeding bullets, since aim correction
     * spoils precisely calculated aim-ahead predictions for projectiles.
     */
    public boolean priHelpAim = true;

    /**
     * Projectile speed of alternate fire mode.
     * Defaults to value 9e+9 for speeding bullets.
     */
    public double altProjectileSpeed = 9e+9;
    /**
     * Max distance to which the weapon can shoot with alternate fire mode.
     * Defaults to 9e+9, when there is no range limit.
     */
    public double altMaxRange = 9e+9;
    /**
     * Splash damage radius of alternate fire mode. Zero if no splash damage.
     */
    public double altSplashRadius = 0;
    /**
     * Whether the weapon is better to be charged, even if no enemy is around.
     */
    public boolean altKeepShooting = false;

    /**
     * Amount of ammo, when the weapon is to be considered empty.
     */
    public int altLowAmmo = 4;

    /**
     * Most effective distance from the enemy while using primary fire mode.
     * This value is used to keep desired and working distance from enemies.
     */
    public double altIdealCombatRange = 300;
    /**
     * Most effective amount of strafing to side.
     */
    public double altStrafingAmount = 300;
    /**
     * Effective distance of alternate fire mode. Beyond this distance, the
     * weapon might start to loose some glance. This value is used as cut-off
     * point for aim-ahead predictions based on enemy distance.
     */
    public double altEffectiveDistance = 9e+9;
    /**
     * How much time it is reasonable to let the projectile of alternate fire
     * mode glide until expected impact. This value is used as cut-off point
     * for aim-ahead predictions based on enemy velocity and distance.
     */
    public double altReasonableFlight = .8;
    /**
     * How much static aim-ahead distance should be used for alternate fire
     * mode. This value is used for aim-ahead predictions of speeding
     * bullets.
     */
    public double altStaticAimAhead = 60;
    /**
     * How much static z-axis correction should be used for alternate fire
     * mode. This value is used for headshots and predictions.
     */
    public double altStaticZCorrection = 0;

    /**
     * Whether server may assist with aim corrections for alternate fire mode.
     * This is reasonable only for speeding bullets, since aim correction
     * spoils precisely calculated aim-ahead predictions for projectiles.
     */
    public boolean altHelpAim = true;

    /*========================================================================*/

    /**
     * Function used for filling-up enum item properties. Used in constructor.
     */
    protected abstract void fill ();

    /**
     * Enum item constructor.
     * @param name UT name of the item.
     * @param title Speech name of the item.
     */
    private LoqueWeaponInfo (String name, String title)
    {
        // set the name and title
        this.name = name;
        this.title = title;
        // fill in some values
        fill ();
    }

    /*========================================================================*/

    /**
     * Decides, whether to use alternate fire mode.
     * @param distance Distance at which the destination is located.
     * Might be -1 if no distance is to be considered while picking the gun.
     * @return Returns true, if alternate mode is recommended.
     */
    public abstract boolean altFire (double distance);

    /**
     * Determines inventory weapon score based on the skill and current ammo.
     * @param skill Skill level of the bot.
     * @param ammo Current amount of ammo.
     * @return Comparable score of the weapon.
     */
    public abstract int inventoryScore (int skill, int ammo);

    /**
     * Determines general weapon score based only on the skill.
     * @param skill Skill level of the bot.
     * @return Comparable score of the weapon.
     */
    public abstract int generalScore (int skill);

    /**
     * Determines combat weapon score based on the skill and target distance.
     * @param skill Skill level of the bot.
     * @param distance Distance, for which the weapon is being chosen.
     * Might be -1 if no distance is to be considered while picking the gun.
     * @return Comparable score of the weapon.
     */
    public abstract int combatScore (int skill, double distance);

    /*========================================================================*/

    /**
     * Chooses among given list of weapons. Considers skill level and distance.
     *
     * <h4>Future</h4>
     *
     * Decission based on enemy weapon is not implemented right now. The main
     * idea is to use (for example) a shield gun against a rocket launcher,
     * since these weapons require different fighting distances and the shield
     * gun might be better choice for engaging in this case, forcing the enemy
     * to rearm or hurt himself while hurting us.
     *
     * @param skill Skill level of the bot.
     * @param weapons List of weapons to choose from.
     * @param enemyDistance Distance, for which the weapon is being chosen.
     * Might be -1 if no distance is to be considered while picking the gun.
     * @param enemyWeapon Weapon, the enemy holds in his hands. Might be null.
     * @return Chosen weapon; or null upon no possible choice.
     */
    public static AddWeapon chooseWeapon (int skill, ArrayList<AddWeapon> weapons, double enemyDistance, String enemyWeapon)
    {
        AddWeapon chosen = null;
        LoqueWeaponInfo chosenInfo = null;

        // run through all weapons
        for (AddWeapon weapon : weapons)
        {
            // is there at least some ammo ready for this weapon?
            if (weapon.currentAmmo <= 0)
                continue;

            // is this a shield gun? no shieldguns here..
            if (weapon.weaponType == ItemType.SHIELD_GUN)
                continue;

            // retreive info about that weapon
            LoqueWeaponInfo weaponInfo = getInfo(weapon.weaponType);

            // is there enough ammo ready for this weapon?
            if (weapon.currentAmmo < weaponInfo.priLowAmmo)
                continue;

            // do we have a weapon to compare it with?
            if (chosen == null)
            {
                chosen = weapon;
                chosenInfo = weaponInfo;
            }
            else
            {
                // is the weapon better for combat?
                if (
                    weaponInfo.combatScore(skill, enemyDistance)
                    > chosenInfo.combatScore(skill, enemyDistance)
                )
                {
                    // oukay then, choose this one..
                    chosen = weapon;
                    chosenInfo = weaponInfo;
                }
            }
        }

        // did we choose a weapon?
        if (chosen != null)
            return chosen;

        // run through all weapons again
        for (AddWeapon weapon : weapons)
        {
            LoqueWeaponInfo weaponInfo = getInfo(weapon.weaponType);

            // is there at least some ammo ready for this weapon?
            if (weapon.currentAmmo <= 0)
                continue;

            // do we have a weapon to compare it with?
            if (chosen == null)
            {
                chosen = weapon;
                chosenInfo = weaponInfo;
            }
            else
            {
                // is the weapon better for combat?
                if (
                    weaponInfo.combatScore(skill, enemyDistance)
                    > chosenInfo.combatScore(skill, enemyDistance)
                )
                {
                    // oukay then, choose this one..
                    chosen = weapon;
                    chosenInfo = weaponInfo;
                }
            }
        }

        // return chosen weapon; or null, if none..
        return chosen;
    }

    /*========================================================================*/

    /**
     * Retreives correct weapon info by given weapon item type.
     * @param itemType Item type of which info is requested.
     * @return Requested weapon info.
     */
    public static LoqueWeaponInfo getInfo (ItemType itemType)
    {
        // return by type
        switch (itemType)
        {
            case SHIELD_GUN: return SHIELD_GUN;
            case ASSAULT_RIFLE: return ASSAULT_RIFLE;
            case BIO_RIFLE: return BIO_RIFLE;
            case SHOCK_RIFLE: return SHOCK_RIFLE;
            case MINIGUN: return MINIGUN;
            case LINK_GUN: return LINK_GUN;
            case FLAK_CANNON: return FLAK_CANNON;
            case ROCKET_LAUNCHER: return ROCKET_LAUNCHER;
            case LIGHTNING_GUN: return LIGHTNING_GUN;
            case SNIPER_RIFLE: return SNIPER_RIFLE;
            case REDEEMER: return REDEEMER;
            default: return UNKNOWN;
        }
    }

    /**
     * Retreives correct weapon info by given weapon name.
     * @param weapon Weapon name of which info is requested.
     * @return Requested weapon info.
     */
    public static LoqueWeaponInfo getInfo (AddWeapon weapon)
    {
        // did we get a weapon?
        if (weapon == null)
            return UNKNOWN;

        // return by type
        return getInfo (weapon.weaponType);
    }

    /**
     * Retreives correct weapon info by given weapon name.
     * @param weapon Weapon name of which info is requested.
     * @return Requested weapon info.
     */
    public static LoqueWeaponInfo getInfo (String weapon)
    {
        // find the appropriate string..
        for (LoqueWeaponInfo info : values())
            if (info.name.compareTo(weapon) == 0)
                return info;

        // return unknown
        return UNKNOWN;
    }

    /*========================================================================*/

    public static void Test ()
    {
        TestWeapon (ASSAULT_RIFLE, 400/*100*/);
        TestWeapon (BIO_RIFLE, 50/*20*/);
        TestWeapon (SHOCK_RIFLE, 50/*20*/);
        TestWeapon (MINIGUN, 300/*150*/);
        TestWeapon (LINK_GUN, 220/*70*/);
        TestWeapon (FLAK_CANNON, 35/*15*/);
        TestWeapon (ROCKET_LAUNCHER, 30/*12*/);
        TestWeapon (LIGHTNING_GUN, 40/*15*/);
        TestWeapon (SNIPER_RIFLE, 35/*15*/);
    }

    public static void TestWeapon (LoqueWeaponInfo w, int ammo)
    {
        int[] skills = {1,2,3,4,5,6,7};

        System.out.printf("%30s", w.name + ": ");

        for (int skill : skills)
            System.out.printf("%4d, ", w.inventoryScore(skill, ammo));

        System.out.println ("ammo " + ammo);
    }

}
